#!/usr/bin/env python3
# coding: Latin-1

# -----------------------------------------------------------------------------
# Project: SolveBOTICS Remote Controller + Receiver with GStreamer Video
# Author: SolveBOTICS LLC
# Website: https://www.solvebotics.com
# Copyright (c) 2025 SolveBOTICS LLC
# License: MIT License
#
# -----------------------------------------------------------------------------
# Attribution Notice
# -----------------------------------------------------------------------------
# This software was originally based on example code by PiBorg
# (https://www.piborg.org/), used under their permissive redistribution terms.
# The original "RemoteKeyBorg" concept and structure inspired this project.
#
# Subsequent modifications, extensions, and enhancements were developed by
# SolveBOTICS LLC, including:
#   - Multi-drive and extended GPIO support
#   - Python 2 and 3 cross-version compatibility
#   - Bidirectional feedback between controller and robot
#   - Integration of real-time GStreamer video pipeline in Pygame
#   - Performance optimizations and modernized architecture
#
# Please retain this attribution in redistributions or derivative works.
#
# -----------------------------------------------------------------------------

from __future__ import print_function
import socket
import time
import pygame
import sys
import threading
import gi
import numpy as np

# -----------------------------------------------------------------------------
# GStreamer setup
# -----------------------------------------------------------------------------
gi.require_version('Gst', '1.0')
from gi.repository import Gst, GLib

Gst.init(None)

PIPELINE_DESC = (
    "udpsrc port=5000 ! "
    "application/x-rtp,media=video,encoding-name=H264,clock-rate=90000,payload=96 ! "
    "rtpjitterbuffer latency=100 ! "
    "rtph264depay ! h264parse ! avdec_h264 ! "
    "videoconvert ! video/x-raw,format=RGB ! appsink name=sink emit-signals=true sync=false"
)

class VideoReceiver(threading.Thread):
    def __init__(self):
        super(VideoReceiver, self).__init__()
        self.daemon = True
        self.pipeline = Gst.parse_launch(PIPELINE_DESC)
        self.appsink = self.pipeline.get_by_name("sink")
        self.appsink.connect("new-sample", self.on_new_sample)
        self.frame = None
        self.running = True

    def on_new_sample(self, sink):
        sample = sink.emit("pull-sample")
        buf = sample.get_buffer()
        caps = sample.get_caps()
        w = caps.get_structure(0).get_value("width")
        h = caps.get_structure(0).get_value("height")
        success, map_info = buf.map(Gst.MapFlags.READ)
        if success:
            frame = np.frombuffer(map_info.data, np.uint8).reshape((h, w, 3))
            #self.frame = np.flipud(frame.copy())  # vertical flip so it's not upside down
            self.frame = np.fliplr(frame.copy())  # horizontal flip so it's not mirrored
            #self.frame = frame.copy()  # vertical flip so it's not upside down
            buf.unmap(map_info)
        return Gst.FlowReturn.OK

    def run(self):
        self.pipeline.set_state(Gst.State.PLAYING)
        bus = self.pipeline.get_bus()
        while self.running:
            msg = bus.timed_pop_filtered(100 * Gst.MSECOND,
                                         Gst.MessageType.ERROR | Gst.MessageType.EOS)
            if msg:
                print("GStreamer message:", msg.type)
                break
        self.pipeline.set_state(Gst.State.NULL)

    def stop(self):
        self.running = False
        self.pipeline.set_state(Gst.State.NULL)

# -----------------------------------------------------------------------------
# Controller settings
# -----------------------------------------------------------------------------
broadcastIP = '192.168.0.107'
broadcastPort = 9038
listenPort = 9039
interval = 0.1
regularUpdate = True

DRIVE1_ENABLE = 1
DRIVE2_L1 = 2
DRIVE3_L4 = 3
DRIVE4_L2 = 4
DRIVE5_L3 = 5
DRIVE6_LIGHT = 6

# -----------------------------------------------------------------------------
# Networking
# -----------------------------------------------------------------------------
sender = socket.socket(socket.AF_INET, socket.SOCK_DGRAM, socket.IPPROTO_UDP)
sender.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)
sender.bind(('0.0.0.0', 0))

receiver = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
receiver.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
receiver.bind(('0.0.0.0', listenPort))
receiver.setblocking(False)

# -----------------------------------------------------------------------------
# Pygame
# -----------------------------------------------------------------------------
pygame.init()
screen = pygame.display.set_mode([640, 480])
pygame.display.set_caption("SolveBOTICS - Video, Control, Feedback")
font = pygame.font.SysFont(None, 24)
clock = pygame.time.Clock()

# -----------------------------------------------------------------------------
# States
# -----------------------------------------------------------------------------
hadEvent = True
moveUp = moveDown = moveLeft = moveRight = moveQuit = False
latestFeedback = "Waiting for feedback..."
lastFeedbackTime = 0.0

# -----------------------------------------------------------------------------
# Input handler
# -----------------------------------------------------------------------------
def PygameHandler(events):
    global hadEvent, moveUp, moveDown, moveLeft, moveRight, moveQuit
    for event in events:
        if event.type == pygame.QUIT:
            moveQuit = True
        elif event.type == pygame.KEYDOWN:
            hadEvent = True
            if event.key == pygame.K_UP:
                moveUp = True
            elif event.key == pygame.K_DOWN:
                moveDown = True
            elif event.key == pygame.K_LEFT:
                moveLeft = True
            elif event.key == pygame.K_RIGHT:
                moveRight = True
            elif event.key == pygame.K_ESCAPE:
                moveQuit = True
        elif event.type == pygame.KEYUP:
            hadEvent = True
            if event.key == pygame.K_UP:
                moveUp = False
            elif event.key == pygame.K_DOWN:
                moveDown = False
            elif event.key == pygame.K_LEFT:
                moveLeft = False
            elif event.key == pygame.K_RIGHT:
                moveRight = False

# -----------------------------------------------------------------------------
# Network helpers
# -----------------------------------------------------------------------------
def send_command(cmd):
    if isinstance(cmd, str):
        cmd = cmd.encode('latin-1')
    sender.sendto(cmd, (broadcastIP, broadcastPort))

# -----------------------------------------------------------------------------
# Feedback listener
# -----------------------------------------------------------------------------
def listen_feedback():
    global latestFeedback, lastFeedbackTime
    while True:
        try:
            data, addr = receiver.recvfrom(1024)
            text = data.decode('latin-1').strip()
            latestFeedback = f"From {addr[0]}: {text}"
            lastFeedbackTime = time.time()
        except socket.error:
            time.sleep(0.05)

threading.Thread(target=listen_feedback, daemon=True).start()

# -----------------------------------------------------------------------------
# Main loop
# -----------------------------------------------------------------------------
video = VideoReceiver()
video.start()

try:
    print("Press [ESC] to quit")

    while True:
        PygameHandler(pygame.event.get())

        if moveQuit:
            break

        if hadEvent or regularUpdate:
            hadEvent = False
            driveCommands = ['X'] * 6

            if moveLeft:
                driveCommands[DRIVE1_ENABLE - 1] = 'ON'
                driveCommands[DRIVE3_L4 - 1] = 'ON'
                driveCommands[DRIVE2_L1 - 1] = 'OFF'
                driveCommands[DRIVE4_L2 - 1] = 'ON'
                driveCommands[DRIVE5_L3 - 1] = 'OFF'
            elif moveRight:
                driveCommands[DRIVE1_ENABLE - 1] = 'ON'
                driveCommands[DRIVE3_L4 - 1] = 'OFF'
                driveCommands[DRIVE2_L1 - 1] = 'ON'
                driveCommands[DRIVE4_L2 - 1] = 'OFF'
                driveCommands[DRIVE5_L3 - 1] = 'ON'
            elif moveUp:
                driveCommands[DRIVE1_ENABLE - 1] = 'ON'
                driveCommands[DRIVE3_L4 - 1] = 'OFF'
                driveCommands[DRIVE2_L1 - 1] = 'OFF'
                driveCommands[DRIVE4_L2 - 1] = 'ON'
                driveCommands[DRIVE5_L3 - 1] = 'ON'
            elif moveDown:
                driveCommands[DRIVE1_ENABLE - 1] = 'ON'
                driveCommands[DRIVE3_L4 - 1] = 'ON'
                driveCommands[DRIVE2_L1 - 1] = 'ON'
                driveCommands[DRIVE4_L2 - 1] = 'OFF'
                driveCommands[DRIVE5_L3 - 1] = 'OFF'
            else:
                for i in range(5):
                    driveCommands[i] = 'OFF'

            command = ','.join(driveCommands)
            send_command(command)

        # ---------------------------------------------------------------------
        # Draw video + feedback
        # ---------------------------------------------------------------------
        if video.frame is not None:
            surf = pygame.surfarray.make_surface(np.rot90(video.frame))
            screen.blit(surf, (0, 0))
        else:
            screen.fill((0, 0, 0))

        # Determine connection color
        time_since_feedback = time.time() - lastFeedbackTime
        if lastFeedbackTime == 0:
            color = (200, 200, 200)
            status = "No data"
        elif time_since_feedback < 2:
            color = (0, 255, 0)
            status = "Connected"
        elif time_since_feedback < 5:
            color = (255, 255, 0)
            status = "Signal weak"
        else:
            color = (255, 0, 0)
            status = "Connection lost"

        # Overlay text
        pygame.draw.rect(screen, (0, 0, 0), (0, 0, 640, 40))  # dark bar
        text1 = font.render("Remote Keyboard Control", True, (255, 255, 255))
        text2 = font.render("Status: " + status, True, color)
        text3 = font.render(latestFeedback, True, color)
        screen.blit(text1, (10, 5))
        screen.blit(text2, (200, 5))
        screen.blit(text3, (10, 420))

        pygame.display.flip()
        clock.tick(30)

    send_command('ALLOFF')

except KeyboardInterrupt:
    send_command('ALLOFF')
finally:
    video.stop()
    pygame.quit()
    sys.exit(0)
