#!/usr/bin/env python
# coding: latin-1
# -----------------------------------------------------------------------------
# Project: SolveBOTICS Remote Controller + Receiver with GStreamer Video
# Author: SolveBOTICS LLC
# Website: https://www.solvebotics.com
# Copyright (c) 2025 SolveBOTICS LLC
# License: MIT License
#
# -----------------------------------------------------------------------------
# Attribution Notice
# -----------------------------------------------------------------------------
# This software was originally based on example code by PiBorg
# (https://www.piborg.org/), used under their permissive redistribution terms.
# The original "RemoteKeyBorg" concept and structure inspired this project.
#
# Subsequent modifications, extensions, and enhancements were developed by
# SolveBOTICS LLC, including:
#   - Multi-drive and extended GPIO support
#   - Python 2 and 3 cross-version compatibility
#   - Bidirectional feedback between controller and robot
#   - Integration of real-time GStreamer video pipeline in Pygame
#   - Performance optimizations and modernized architecture
#
# Please retain this attribution in redistributions or derivative works.
#
# -----------------------------------------------------------------------------

from __future__ import print_function
import sys, socket, threading, time

# --- Compatibility imports for Python 2 and 3 ---
try:
    import socketserver          # Python 3
except ImportError:
    import SocketServer as socketserver  # Python 2

try:
    input = raw_input            # Python 2 compatibility
except NameError:
    pass                         # Python 3 already has input()

import RPi.GPIO as GPIO
GPIO.setmode(GPIO.BCM)

# --- Pin assignments ---
DRIVE_1 = 5  # ENA, ENB of L298 driver PCB, connect to Pin #29
DRIVE_2 = 4  # IN1 of L298 PCB, connect to Pin #7 of Pi GPIO header
DRIVE_3 = 18 # IN4 of L298 PCB, connect to Pin #12 of Pi GPIO header
DRIVE_4 = 7  # IN2 of L298 PCB, connect to Pin #26 of Pi GPIO header
DRIVE_5 = 8  # IN3 of L298 PCB, connect to Pin #24 of Pi GPIO header
DRIVE_6 = 9  # Extra signal, not currently assigned, Pin #21

# ENA and ENB of L298 PCB jumpered high, so DRIVE_1 may not be necessary

lDrives = [DRIVE_1, DRIVE_2, DRIVE_3, DRIVE_4, DRIVE_5, DRIVE_6]

# --- Setup pins as outputs ---
for pin in lDrives:
    GPIO.setup(pin, GPIO.OUT)

# --- Helper function ---
def MotorOff():
    for drive in lDrives:
        GPIO.output(drive, GPIO.LOW)

# --- UDP network settings ---
portListen = 9038          # Receive control commands
feedbackPort = 9039        # Send feedback/heartbeat
feedbackInterval = 1.0     # Seconds between feedback packets

# --- Feedback sender thread ---
class FeedbackThread(threading.Thread):
    def __init__(self, stop_event):
        threading.Thread.__init__(self)
        self.daemon = True
        self.stop_event = stop_event
        self.sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        self.sock.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)

    def run(self):
        while not self.stop_event.is_set():
            try:
                # Build a simple feedback string
                msg = "STATUS:OK,TIME:{:.1f}".format(time.time())
                # Send to all (broadcast)
                self.sock.sendto(msg.encode('latin-1'), ('<broadcast>', feedbackPort))
            except Exception as e:
                print("Feedback error:", e)
            time.sleep(feedbackInterval)

    def close(self):
        self.stop_event.set()
        self.sock.close()

# --- UDP command handler ---
class R22Handler(socketserver.BaseRequestHandler):
    def handle(self):
        global isRunning
        request, sock = self.request

        if isinstance(request, bytes):
            request = request.decode('latin-1')

        request = request.strip().upper()
        driveCommands = request.split(',')

        if len(driveCommands) == 1:
            # Special single-word commands
            if request == 'ALLOFF':
                MotorOff()
                print('All drives off')
            elif request == 'EXIT':
                isRunning = False
            else:
                print('Special command "{}" not recognised'.format(request))
        elif len(driveCommands) == len(lDrives):
            for driveNo, command in enumerate(driveCommands):
                command = command.strip()
                if command == 'ON':
                    GPIO.output(lDrives[driveNo], GPIO.HIGH)
                elif command == 'OFF':
                    GPIO.output(lDrives[driveNo], GPIO.LOW)
                elif command == 'X':
                    pass
                else:
                    print('Drive {} command "{}" not recognised!'.format(driveNo, command))
        else:
            print('Command "{}" did not have {} parts!'.format(request, len(lDrives)))

# --- Main program ---
try:
    global isRunning
    MotorOff()
    print('Running...')

    remoteR22server = socketserver.UDPServer(('', portListen), R22Handler)
    isRunning = True

    # Start feedback sender
    stopEvent = threading.Event()
    feedbackThread = FeedbackThread(stopEvent)
    feedbackThread.start()
    print("Feedback thread started on UDP port {}".format(feedbackPort))

    while isRunning:
        remoteR22server.handle_request()

    print('Finished')
    stopEvent.set()
    feedbackThread.join(timeout=2.0)
    MotorOff()
    input('Turn the power off now, press ENTER to continue')
    GPIO.cleanup()

except KeyboardInterrupt:
    print('Terminated')
    try:
        stopEvent.set()
        feedbackThread.join(timeout=2.0)
    except Exception:
        pass
    MotorOff()
    GPIO.cleanup()
